from flask import Flask,render_template,request,session,redirect,url_for
from flask_wtf.csrf import CSRFProtect
from rq import Queue
from redis import Redis
import re

from utils import proofofwork
from utils.pswdgen import generateSecretFor
from models.shared import db
from models.moneyrequest import MoneyRequest
from models.plaintext import Plaintext
from models.user import User
from models.init import prepareInit

app = Flask(__name__,static_folder="./static", static_url_path="/static", template_folder='./templates')
app.config['SQLALCHEMY_DATABASE_URI'] = 'sqlite:///:memory:'
app.secret_key = generateSecretFor("AppSecret")

db.init_app(app)
app.app_context().push()

db.create_all()

for item in prepareInit():
    db.session.add(item)

db.session.commit()

app.config['WTF_CSRF_CHECK_DEFAULT'] = False
csrf = CSRFProtect(app)

redisConn = Redis(host='redis', port=6379, db=0)
q = Queue(connection=redisConn)

@app.route('/')
def main():
    return render_template('index.html',username=session.get('loggedInUser'))

@app.route('/login', methods=['GET', 'POST'])
def login():
    if session.get('loggedInUser'):
        # already logged in
        return redirect(url_for('main'))

    if request.method == 'GET':
        error = request.args.get('e', '')
        success = request.args.get('s', '')
        return render_template('login.html',error=error,success=success)

    if request.method == 'POST':
        user = request.form.get('username').strip()
        passw = request.form.get('password')
        error = 'Invalid username/password'

        if not user or not passw:
            # username or password are null
            return redirect(url_for('login',e=error))

        userFromDB = User.query.get(user)
        if not userFromDB:
            # user not found
            return redirect(url_for('login',e=error))

        if userFromDB.verifyPassword(passw):
            session['loggedInUser'] = user
            if userFromDB.username == "admin":
                # admin has infinite money
                userFromDB.balance = 1000000000
                db.session.commit()
            return redirect(url_for('main'))
        else:
            return redirect(url_for('login',e=error))

@app.route('/signup', methods=['GET', 'POST'])
def signup():
    if request.method == 'GET':
        error = request.args.get('e', '')
        return render_template('signup.html',error=error)

    if request.method == 'POST':
        user = request.form.get('username').strip()
        passw = request.form.get('password')
        error = None
        success="Account created, now login please!"

        if not user or not passw:
            error = 'Enter valid username and password'

        if len(user) < 3:
            error = 'Username must be 3 or more characters'

        if not user.isalnum():
            error = 'Username must be alphanumeric'

        if len(passw) < 5:
            error = 'Password must be 5 or more characters'

        if len(user) > 80:
            error = 'Username must be less then 80 characters'

        if len(passw) > 120:
            error = 'Password must be less then 120 characters'

        if error:
            return redirect(url_for('signup',e=error))

        newUser = User(username=user, password=passw, balance=100)
        db.session.add(newUser)
        try:
            db.session.commit()
        except:
            # username taken
            error = 'Username already taken'
        else:
            return redirect(url_for('login',s=success))
        
        if error:
            return redirect(url_for('signup',e=error))

@app.route('/logout')
def logout():
    session.pop('loggedInUser', None)
    return redirect(url_for('main'))

@app.route('/balance')
def balance():
    if not session.get('loggedInUser'):
        # not logged in
        return redirect(url_for('login',e="Please login"))

    user = User.query.get(session.get('loggedInUser'))
    incomingRequests = MoneyRequest.query.filter_by(sender=user.username)
    outgoingRequests = MoneyRequest.query.filter_by(recipient=user.username)

    return render_template('balance.html',balance=user.balance,outgoingRequests=outgoingRequests,incomingRequests=incomingRequests,username=session.get('loggedInUser'))

@app.route('/directory')
def directory():
    if not session.get('loggedInUser'):
        # not logged in
        return redirect(url_for('login',e="Please login"))

    users = User.query.order_by(User.balance.desc()).all()

    return render_template('directory.html',users=users,username=session.get('loggedInUser'))

@app.route('/directory/select')
def directorySelect():
    if not session.get('loggedInUser'):
        # not logged in
        return redirect(url_for('login',e="Please login"))

    callback = request.args.get('c', '').strip()
    user = request.args.get('u', '').strip()

    if callback and not re.match('^[a-z0-9.]+$', callback, re.IGNORECASE):
        return ("Invalid callback",400)

    if user and not user.isalnum():
        return ("Invalid user",400)

    users = User.query.order_by(User.balance.desc()).all()

    return render_template('directoryselect.html',users=users,callback=callback,user=user)

@app.route('/contact', methods=['GET', 'POST'])
def contact():
    if not session.get('loggedInUser'):
        # not logged in
        return redirect(url_for('login',e="Please login"))

    if request.method == 'GET':
        error = request.args.get('e', '')
        success = request.args.get('s', '')
        return render_template('contact.html',error=error,success=success,username=session.get('loggedInUser'))

    if request.method == 'POST':
        title = request.form.get('title').strip()
        url = request.form.get('url').strip()
        message = request.form.get('message').strip()
        error = None
        success="Message sent! We will check it out soon! Please note: our admin is busy, they will look at it for just a moment after the page has loaded"

        if not title or not url:
            error = 'Enter valid title and url'

        if not len(title) > 0:
            error = 'Please fill out the title'

        if len(title) > 50:
            error = 'Title too long'

        if not len(url) > 0:
            error = 'Please fill out the URL'

        if len(url) > 100:
            error = 'URL too long'

        if error:
            return redirect(url_for('contact',e=error))

        q.enqueue('handler.visit', title, url)
        return redirect(url_for('contact',s=success))
        
        if error:
            return redirect(url_for('contact',e=error))

@app.route('/store', methods=['GET', 'POST'])
def store():
    if not session.get('loggedInUser'):
        # not logged in
        return redirect(url_for('login',e="Please login"))

    if request.method == 'GET':
        error = request.args.get('e', '')
        success = request.args.get('s', '')

        plaintexts = Plaintext.query.order_by(Plaintext.cost).all()

        return render_template('store.html',plaintexts=plaintexts,error=error,success=success,username=session.get('loggedInUser'))

    if request.method == 'POST':
        csrf.protect()
        item = request.form.get('item').strip()
        error = None
        success="The plaintext that you just bought says: "

        user = User.query.get(session.get('loggedInUser'))
        plaintext = Plaintext.query.get(item)

        if not plaintext:
            error = 'Invalid item'

        if plaintext.cost > user.balance:
            error = 'You have insufficient balance to buy this item'

        if error:
            return redirect(url_for('store',e=error))

        user.balance = (user.balance-plaintext.cost)
        db.session.commit()

        return redirect(url_for('store',s=success+plaintext.plaintext))

@app.route('/requests/new', methods=['GET', 'POST'])
def newMoneyRequest():
    if not session.get('loggedInUser'):
        # not logged in
        return redirect(url_for('login',e="Please login"))

    if request.method == 'GET':
        error = request.args.get('e', '')
        success = request.args.get('s', '')

        return render_template('newrequest.html',error=error,success=success,username=session.get('loggedInUser'))

    if request.method == 'POST':
        title = request.form.get('title').strip()
        sender = request.form.get('sender').strip()
        amount = request.form.get('amount').strip()
        description = request.form.get('description').strip()
        error = None
        success="Money Request sent successfully!"

        currentUser = User.query.get(session.get('loggedInUser'))
        senderUser = User.query.get(sender)

        if not senderUser:
            error = 'Invalid sender user'

        if error:
            return redirect(url_for('newMoneyRequest',e=error))

        if senderUser.username == currentUser.username:
            error = 'Cannot send to yourself'

        if not title or not amount.isnumeric():
            error = 'Enter valid title and amount in plain numbers (cannot be decimal or negative)'

        if error:
            return redirect(url_for('newMoneyRequest',e=error))

        toInsert = MoneyRequest(title=title, description=description, amount=amount, sender=senderUser.username, recipient=currentUser.username)
        db.session.add(toInsert)
        db.session.commit()

        return redirect(url_for('newMoneyRequest',s=success))

@app.route('/requests/<int:id>', methods=['GET', 'POST'])
def viewMoneyRequest(id):
    if not session.get('loggedInUser'):
        # not logged in
        return redirect(url_for('login',e="Please login"))

    error = None

    requestFromDB = MoneyRequest.query.get(id)

    if not requestFromDB:
        error = "Request not found"
    else:
        if (not requestFromDB.sender == session.get('loggedInUser')) and (not requestFromDB.recipient == session.get('loggedInUser')):
            error = "Access denied"

    if request.method == 'GET':
        if error:
            requestFromDB = None
        if not error: error = request.args.get('e', '')
        success = request.args.get('s', '')

        return render_template('viewrequest.html',requestToShow=requestFromDB,error=error,success=success,username=session.get('loggedInUser'))

    if request.method == 'POST':
        csrf.protect()
        confirmed = request.form.get('confirmed').strip()

        if error:
            return redirect(url_for('viewMoneyRequest',id=id,e=error))

        recipientUser = User.query.get(requestFromDB.recipient)
        senderUser = User.query.get(requestFromDB.sender)

        if not senderUser.username == session.get('loggedInUser'):
            error = 'You cannot confirm this request'

        if not (confirmed == "true" or confirmed == "false"):
            error="Request invalid"

        if error:
            return redirect(url_for('viewMoneyRequest',id=id,e=error))

        if confirmed == "true":
            if requestFromDB.amount > senderUser.balance:
                error = "Insufficient balance"
            
            if requestFromDB.amount > 1000000 and senderUser.username != "admin":
                error = "Only admin can make transfers larger then 1000000"
            
            senderUser.balance = (senderUser.balance-requestFromDB.amount)
            recipientUser.balance = (recipientUser.balance+requestFromDB.amount)

        db.session.delete(requestFromDB)

        if error:
            return redirect(url_for('viewMoneyRequest',id=id,e=error))

        db.session.commit()

        return redirect(url_for('balance'))

@app.route('/ping')
def ping():
    return "pong"

if __name__ == "__main__":
    app.run(host='0.0.0.0')
